/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.labs.jaxmas.registry;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.JAXRException;
import javax.xml.registry.LifeCycleManager;
import javax.xml.registry.infomodel.Association;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.RegistryEntry;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Slot;

import org.apache.labs.jaxmas.registry.accessor.ROAccessors;
import static org.junit.Assert.*;
import org.junit.Test;


/**
 * Test case for creating a taxonomy and iterating over it.
 */
public class TaxonomiesTestCase extends AbstractInitializedJaxMasTestCase {
	private static final String TEST_CLASSIFICATION_SCHEME_DESCRIPTION = "Test Classification Scheme Description"; //$NON-NLS-1$
	private static final String TEST_CLASSIFICATION_SCHEME_NAME = "Test Classification Scheme"; //$NON-NLS-1$
	private static final String TEST_CONCEPT_NAME = "Test Concept"; //$NON-NLS-1$
	private static final String TEST_CONCEPT_VALUE = "Test Concept Value"; //$NON-NLS-1$
	private static final String TEST_SLOT1_NAME = "Test Slot 1"; //$NON-NLS-1$
    private static final String TEST_SLOT2_NAME = "Test Slot 2"; //$NON-NLS-1$

	/**
	 * Tests creating a classification scheme.
	 */
	@Test public void testCreateClasssificationScheme() throws Exception {
        removeExistingClassificationScheme();
        final ClassificationScheme cs = createClassificationScheme();
		deleteClassificationScheme(cs);
	}

	private void deleteConcept(final Concept pConcept) throws JAXRException {
        for (Object o : pConcept.getChildrenConcepts()) {
            deleteConcept((Concept) o);
        }
        final Connection connection = getConnection();
        final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
        blm.deleteConcepts(Collections.singleton(pConcept.getKey()));
	}

	private void deleteClassificationScheme(final ClassificationScheme pClassificationScheme)
            throws JAXRException {
        for (Object o : pClassificationScheme.getChildrenConcepts()) {
            deleteConcept((Concept) o);
        }
        final Connection connection = getConnection();
        final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
        final BusinessQueryManager bqm = connection.getRegistryService().getBusinessQueryManager();
        blm.deleteClassificationSchemes(Collections.singleton(pClassificationScheme.getKey()));
        final ClassificationScheme cs = bqm.findClassificationSchemeByName(null, TEST_CLASSIFICATION_SCHEME_NAME);
        assertNull(cs);
    }

	private ClassificationScheme createClassificationScheme() throws JAXRException {
	    final Connection connection = getConnection();
	    final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
	    final ClassificationScheme cs = blm.createClassificationScheme(newString(TEST_CLASSIFICATION_SCHEME_NAME), newString(TEST_CLASSIFICATION_SCHEME_DESCRIPTION));
        blm.saveClassificationSchemes(Collections.singleton(cs));
        final BusinessQueryManager bqm = connection.getRegistryService().getBusinessQueryManager();
        final ClassificationScheme cs2 = bqm.findClassificationSchemeByName(null, TEST_CLASSIFICATION_SCHEME_NAME);
        assertNotNull(cs2);
        assertEquals(TEST_CLASSIFICATION_SCHEME_NAME, asString(cs.getName()));
        assertEquals(TEST_CLASSIFICATION_SCHEME_DESCRIPTION, asString(cs.getDescription()));
        return cs;
	}

	private void removeExistingConcepts(Collection<?> pConcepts) throws JAXRException {
        final Connection connection = getConnection();
        final List<Key> keys = new ArrayList<Key>();
        final BusinessQueryManager bqm = connection.getRegistryService().getBusinessQueryManager();
        final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
        for (Object o : pConcepts) {
            final Concept concept = (Concept) o;
            for (Object o2 : bqm.findAssociations(null, null, concept.getKey().getId(), null).getCollection()) {
                final Association assoc = (Association) o2;
                blm.deleteAssociations(Collections.singleton(assoc.getKey()));
            }
            keys.add(concept.getKey());
        }
        blm.deleteConcepts(keys);
	}
	
    private void removeExistingClassificationScheme()
            throws JAXRException {
        final Connection connection = getConnection();
        final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
        final BusinessQueryManager bqm = connection.getRegistryService().getBusinessQueryManager();
        final BulkResponse br = bqm.findClassificationSchemes(null, Collections.singleton(TEST_CLASSIFICATION_SCHEME_NAME), null, null);
        final Collection<?> oldCsCollection = br.getCollection();
        if (oldCsCollection != null  &&  !oldCsCollection.isEmpty()) {
            final List<Key> keys = new ArrayList<Key>();
            for (Object o : oldCsCollection) {
                final ClassificationScheme cs = (ClassificationScheme) o;
                removeExistingConcepts(cs.getChildrenConcepts());
                keys.add(cs.getKey());
            }
            blm.deleteClassificationSchemes(keys);
        }
    }

    /**
     * Tests creating a concept.
     */
    @Test public void testCreateConcepts() throws Exception {
        removeExistingClassificationScheme();
        final Concept concept = createConcept();
        final Concept otherConcept = loadConcept();
        assertNotNull(otherConcept);
        assertEquals(otherConcept.getKey(), concept.getKey());
        assertEquals(TEST_CONCEPT_NAME, asString(otherConcept.getName()));
        deleteClassificationScheme(concept.getClassificationScheme());
        final Concept nullConcept = loadConcept();
        assertNull(nullConcept);
    }

    private Concept loadConcept() throws JAXRException {
        return getConceptByPath(TEST_CLASSIFICATION_SCHEME_NAME + "/" + TEST_CONCEPT_VALUE); //$NON-NLS-1$
    }

    private Concept createConcept() throws JAXRException {
        final ClassificationScheme cs = createClassificationScheme();
        final Connection connection = getConnection();
        final BusinessLifeCycleManager blm = connection.getRegistryService().getBusinessLifeCycleManager();
        final Concept concept = blm.createConcept(cs, newString(TEST_CONCEPT_NAME), TEST_CONCEPT_VALUE);
        assertNotNull(concept);
        blm.saveConcepts(Collections.singleton(concept));
        return concept;
    }

    private void checkSlot(RegistryObject pRegistryObject, String pSlotName, String... pValues) throws JAXRException {
        final Slot slot = pRegistryObject.getSlot(pSlotName);
        assertNotNull(slot);
        checkSlot(slot, pSlotName, pValues);
    }

    private void checkSlot(Slot pSlot, String pSlotName, String... pValues) throws JAXRException {
        assertEquals(pSlotName, pSlot.getName());
        final Collection<?> values = pSlot.getValues();
        assertEquals(pValues.length, values.size());
        int i = 0;
        for (Object o : values) {
            assertEquals(pValues[i++], (String) o);
        }
        assertEquals(pValues.length, i);
    }
    
    /**
     * Tests adding slots to a concept.
     */
    @Test public void testConceptSlots() throws Exception {
        removeExistingClassificationScheme();
        final Concept concept = createConcept();
        final Slot slot1 = getBusinessLifecycleManager().createSlot(TEST_SLOT1_NAME, "v1", null); //$NON-NLS-1$
        concept.addSlot(slot1);
        final Slot slot2 = getBusinessLifecycleManager().createSlot(TEST_SLOT2_NAME, Arrays.asList(new String[]{"v2", "v3"}), null); //$NON-NLS-1$ //$NON-NLS-2$
        concept.addSlot(slot2);
        getBusinessLifecycleManager().saveObjects(Collections.singleton(concept));
        final Concept otherConcept = loadConcept();
        final Iterator<?> slots = otherConcept.getSlots().iterator();
        assertTrue(slots.hasNext());
        Slot otherSlot1 = (Slot) slots.next();
        assertTrue(slots.hasNext());
        Slot otherSlot2 = (Slot) slots.next();
        assertFalse(slots.hasNext());
        if (TEST_SLOT2_NAME.equals(otherSlot1.getName())) {
            final Slot slot = otherSlot1;
            otherSlot1 = otherSlot2;
            otherSlot2 = slot;
        }
        checkSlot(otherSlot1, TEST_SLOT1_NAME, "v1"); //$NON-NLS-1$
        checkSlot(otherSlot2, TEST_SLOT2_NAME, "v2", "v3"); //$NON-NLS-1$ //$NON-NLS-2$
        checkSlot(otherConcept, TEST_SLOT1_NAME, "v1"); //$NON-NLS-1$
        checkSlot(otherConcept, TEST_SLOT2_NAME, "v2", "v3");  //$NON-NLS-1$//$NON-NLS-2$
        deleteClassificationScheme(concept.getClassificationScheme());
    }

    /**
     * Tests the presence of object types.
     */
    @Test public void testObjectTypesPresent() throws Exception {
        removeExistingClassificationScheme();
        final Concept concept = createConcept();
        final Concept conceptType = concept.getObjectType();
        assertNotNull(conceptType);
        assertEquals(conceptType.getKey(), conceptType.getObjectType().getKey());
        assertEquals(conceptType.getKey().getId(), ROAccessors.ObjectTypes.CONCEPT.getId());
        assertEquals("Category", conceptType.getValue()); //$NON-NLS-1$
        final Concept clSchemeType = concept.getClassificationScheme().getObjectType();
        assertNotNull(clSchemeType);
        assertEquals(clSchemeType.getKey().getId(), ROAccessors.ObjectTypes.CLASSIFICATION_SCHEME.getId());
        assertEquals("Taxonomy", clSchemeType.getValue()); //$NON-NLS-1$
        deleteClassificationScheme(concept.getClassificationScheme());
    }

    /**
     * Tests the creation of classifications.
     */
    @Test public void testClassifications() throws Exception {
        removeExistingClassificationScheme();
        final Concept concept = createConcept();
        final BusinessLifeCycleManager blcm = getBusinessLifecycleManager();
		blcm.saveObjects(Collections.singleton(concept));
        final RegistryEntry re = (RegistryEntry) blcm.createObject(LifeCycleManager.REGISTRY_ENTRY);
        assertNotNull(re);
        final Classification cl = blcm.createClassification(concept);
        assertNotNull(cl);
        re.addClassification(cl);
        assertEquals(concept.getValue(), cl.getValue());
        assertEquals(concept.getKey(), cl.getConcept().getKey());
        assertEquals(concept.getClassificationScheme().getKey(), cl.getClassificationScheme().getKey());
        blcm.saveObjects(Collections.singleton(re));
        final RegistryEntry reOther = (RegistryEntry) getBusinessQueryManager().getRegistryObject(re.getKey().getId());
        assertNotNull(reOther);
        boolean found = false;
        for (Object o : reOther.getClassifications()) {
            final Classification clOther = (Classification) o;
            if (clOther.getKey().equals(cl.getKey())) {
                assertEquals(concept.getKey(), cl.getConcept().getKey());
                assertEquals(concept.getClassificationScheme().getKey(), cl.getClassificationScheme().getKey());
                assertEquals(concept.getValue(), cl.getValue());
                found = true;
                break;
            }
        }
        assertTrue(found);
        blcm.deleteObjects(Collections.singleton(re.getKey()));
        deleteClassificationScheme(concept.getClassificationScheme());
    }

    /**
     * Tests the creation of associations.
     */
    @Test public void testAssociations() throws Exception {
        removeExistingClassificationScheme();
        final Concept concept = createConcept();
        getBusinessLifecycleManager().saveObjects(Collections.singleton(concept));
        final RegistryEntry re = (RegistryEntry) getBusinessLifecycleManager().createObject(LifeCycleManager.REGISTRY_ENTRY);
        assertNotNull(re);
        final Concept belongsTo = getBusinessQueryManager().findConceptByPath("AssociationType/BelongsTo"); //$NON-NLS-1$
        assertNotNull(belongsTo);
        final Association assoc = getBusinessLifecycleManager().createAssociation(concept, belongsTo);
        assertNotNull(assoc);
        re.addAssociation(assoc);
        assertEquals(assoc.getSourceObject(), re);
        assertEquals(assoc.getAssociationType(), belongsTo);
        assertEquals(assoc.getTargetObject(), concept);
        getBusinessLifecycleManager().saveObjects(Collections.singleton(re));
        final RegistryEntry reOther = (RegistryEntry) getBusinessQueryManager().getRegistryObject(re.getKey().getId());
        assertNotNull(reOther);
        boolean found = false;
        for (Object o : reOther.getAssociations()) {
            final Association assocOther = (Association) o;
            if (assocOther.getKey().equals(assoc.getKey())) {
                assertEquals(assoc.getSourceObject(), assocOther.getSourceObject());
                assertEquals(assoc.getAssociationType(), assocOther.getAssociationType());
                assertEquals(assoc.getTargetObject(), assocOther.getTargetObject());
                found = true;
                break;
            }
        }
        assertTrue(found);
        getBusinessLifecycleManager().deleteObjects(Collections.singleton(re.getKey()));
        deleteClassificationScheme(concept.getClassificationScheme());
    }
}
